import ij.*;
import ij.plugin.*;
import ij.process.*;
import ij.measure.ResultsTable;
import ij.gui.GenericDialog;

/* Measures red and green values, unmasked and masked, for an edited 4D hyperstack. */
public class Analyze_Edited_Movie implements PlugIn {
  
	private ImagePlus hyperStack;
	private int frames, slices, width, height;                         // Hyperstack parameters.
	private double interval;
	private String title;
	private double[] redValues, greenValues, redValuesMasked, greenValuesMasked;
	private ResultsTable results = new ResultsTable();
	private ImagePlus red, green, redMask, greenMask;
	
	//------------------------------------------------------------------------------------------------------------------------  
	
    public void run(String arg) {

      hyperStack = IJ.getImage();
      title = hyperStack.getTitle();
 
      if ( !(hyperStack.getBitDepth() == 8)  || !(title.endsWith(".tif") || title.endsWith(".TIF")) || 
          !hyperStack.isHyperStack() || !(hyperStack.getNSlices() > 1) ) {
        IJ.showMessage("This plugin requires an 8-bit TIFF hyperstack.");
        hyperStack.close();
        return;
      }
      
      title = title.substring(0, title.length() - 4);                   // Remove the extension.
     
      GenericDialog gd = new GenericDialog("Z-Stack Interval");
      gd.addNumericField("Z-Stack Interval:", 2.00, 2);                 // The default interval is 2.00 sec.
      gd.showDialog();
      if (gd.wasCanceled()) return;
      interval = gd.getNextNumber();
      
      width = hyperStack.getWidth();
      height = hyperStack.getHeight();
      slices = hyperStack.getNSlices();
      frames = hyperStack.getNFrames();
      
      redValues = new double[frames];
      greenValues = new double[frames];
      redValuesMasked = new double[frames];
      greenValuesMasked = new double[frames];
      
      // Create images to hold the red and green data, and the corresponding binary masks.
      red = IJ.createImage("Red", "8-bit black", width, height, 1);
      green = IJ.createImage("Green", "8-bit black", width, height, 1);
      redMask = IJ.createImage("Red Mask", "8-bit black", width, height, 1);
      greenMask = IJ.createImage("Green Mask", "8-bit black", width, height, 1);
      
      // For each time point, sum the mean values for the red and green images in each slice.
      // Then do the same for each color after applying a mask for the opposite color.
      for (int t = 1; t <= frames; t++) {
        for (int z = 1; z <= slices; z++) {
          hyperStack.setPositionWithoutUpdate(1, z, t);
          red.getProcessor().copyBits(hyperStack.getProcessor(), 0, 0, Blitter.COPY);
          redMask.getProcessor().copyBits(hyperStack.getProcessor(), 0, 0, Blitter.COPY);
          redMask.getProcessor().threshold(0);
          redValues[t - 1] += red.getStatistics().mean;

          hyperStack.setPositionWithoutUpdate(2, z, t); 
          green.getProcessor().copyBits(hyperStack.getProcessor(), 0, 0, Blitter.COPY);
          greenMask.getProcessor().copyBits(hyperStack.getProcessor(), 0, 0, Blitter.COPY);
          greenMask.getProcessor().threshold(0);
          greenValues[t - 1] += green.getStatistics().mean;
          
          // Now trim the red and green images using the opposite color masks.
          red.getProcessor().copyBits(greenMask.getProcessor(), 0, 0, Blitter.AND);
          redValuesMasked[t - 1] += red.getStatistics().mean;
          
          green.getProcessor().copyBits(redMask.getProcessor(), 0, 0, Blitter.AND);
          greenValuesMasked[t - 1] += green.getStatistics().mean;
        }
        
        results.incrementCounter();
        results.addValue("Time", interval * (t - 1));
        results.addValue("Red", redValues[t - 1]);
        results.addValue("Green", greenValues[t - 1]);
        results.addValue("Red Masked", redValuesMasked[t - 1]);
        results.addValue("Green Masked", greenValuesMasked[t - 1]);
      }
      
      red.close();
      green.close();
      redMask.close();
      greenMask.close();
      
      //results.showRowNumbers(false);
      results.show(title);
      
      hyperStack.setPosition(1,1,1);
	}

}
